#!/opt/imh-python/bin/python3
import time

import psutil
from rads import send_email
from platform import node


KNOWN_MALICIOUS_CMDLINE_STRINGS = [
    'httpd.pl',
    'bash',
    'exim',
    'proc',
    './cache.sh',
    './xmr',
    'xargsu',
    'perxg',
    'mdxfs',
    './backupm',
    './dirty',
    './apache2',
    '/usr/bin/host',
    '/usr/sbin/acpid',
    './cron.php',
    './milemined',
    './annizod',
    './fpm-worker-main',
    '[stealth]',
]
KNOWN_NONMALICOUS_CMDLINE_STRINGS = [
    'usr/local/cpanel/bin/ftpput',
    '/usr/local/cpanel/3rdparty/bin/awstats.pl',
    'mail.cgi',
]


def get_system_processes():
    """
    Use psutil to generate a list of all system processes
    :return: list of all system processes
    """
    all_system_processes = []

    for system_process in psutil.process_iter():
        try:
            process_info = system_process.as_dict(
                attrs=['username', 'pid', 'ppid', 'create_time', 'cmdline']
            )
        except psutil.NoSuchProcess:
            pass
        else:
            all_system_processes.append(process_info)

    return all_system_processes


def filter_processes(all_system_processes):
    """
    Use some criteria to filter out the malicious processes and create a list
    of them
     - ppid is 1
     - username is not root (non-root)
     - process is older than 5 minutes
     - process 'cmdline' string matches known malicious cmdline string
     - process 'cmdline' string doesn't match known nonmalicious cmdline string
    :param list of all system processes
    :return: list of only malicious system processes
    """
    now = time.time()
    darkmailer_processes = []
    for system_process in all_system_processes:
        seconds = now - system_process['create_time']
        if (
            system_process['ppid'] == 1
            and system_process['username'] != "root"
            and seconds > 300
            and system_process["cmdline"][0] in KNOWN_MALICIOUS_CMDLINE_STRINGS
            and system_process['cmdline'][0]
            not in KNOWN_NONMALICOUS_CMDLINE_STRINGS
        ):
            darkmailer_processes.append(system_process)

    return darkmailer_processes


def main():
    """
    Function that manages flow of nagios check
     - Find all system processes
     - Filter the malicious processes
     - Generate nagios exit status
    """
    all_system_processes = get_system_processes()
    darkmailer_processes = filter_processes(all_system_processes)

    if darkmailer_processes:
        # build email body message
        darkmailer_processes_crit_data = []
        for darkmailer_process in darkmailer_processes:
            darkmailer_processes_crit_data.append(
                (darkmailer_process['pid'], darkmailer_process['username'])
            )

        body = "{} CRITICAL: {} malicious scripts: {}".format(
            node(),
            len(darkmailer_processes_crit_data),
            darkmailer_processes_crit_data,
        )

        send_email(
            'str@imhadmin.net',
            "Darkmailer Processes",
            body,
            html=None,
            sender=None,
            ssl=False,
            server=('localhost', 0),
            login=None,
        )


if __name__ == "__main__":
    main()
