"""Abstract parent class for check_software CMS modules"""
from abc import abstractmethod
from io import StringIO
from pathlib import Path
from typing import Union
import rads


class BleachedColors:
    """Mimics rads.color, except all functions return input unchanged"""

    @staticmethod
    def red(data: str):
        return data

    @staticmethod
    def green(data: str):
        return data

    @staticmethod
    def bold(data: str):
        return data

    @staticmethod
    def yellow(data: str):
        return data

    @staticmethod
    def blue(data: str):
        return data


class HtmlColors:
    """Mimics rads.color, except using HTML"""

    @staticmethod
    def red(data: str):
        return f'<span style="color: #FF0000">{data}</span>'

    @staticmethod
    def green(data: str):
        return f'<span style="color: #008000">{data}</span>'

    @staticmethod
    def bold(data: str):
        return f'<span style="font-weight: bold">{data}</span>'

    @staticmethod
    def yellow(data: str):
        # actually uses orange; yellow on white background is unreadable
        return f'<span style="color: #FF8C00">{data}</span>'

    @staticmethod
    def blue(data: str):
        return f'<span style="color: #0000FF">{data}</span>'


class ModTemplate:
    def __init__(self, args, path: Path, susp: bool, kb_urls: dict[str, str]):
        self.out = StringIO()
        self.args = args
        self.kb_urls = kb_urls
        if self.args.style == 'str':
            self.color = HtmlColors()
        elif self.args.style == 'bleach':
            self.color = BleachedColors()
        else:
            self.color = rads.color
        self.bold(f' == {self.cms_name}: {path} == ')
        if susp:
            self.red('User is currently suspended')
        self.scan_install(path)

    def urlize(self, url: str, label: Union[str, None]) -> str:
        assert self.args.style == 'str'
        if label:
            return f'<a href="{url}">{label}</a>'
        return f'<a href="{url}">{url}</a>'

    def green(self, msg: str, sep=' ', end='\n'):
        print(self.color.green(msg), file=self.out, sep=sep, end=end)

    def bold(self, msg: str, sep=' ', end='\n'):
        print(self.color.bold(msg), file=self.out, sep=sep, end=end)

    def red(self, msg: str, sep=' ', end='\n'):
        print(self.color.red(msg), file=self.out, sep=sep, end=end)

    def yellow(self, msg: str, sep=' ', end='\n'):
        print(self.color.yellow(msg), file=self.out, sep=sep, end=end)

    def blue(self, msg: str, sep=' ', end='\n'):
        print(self.color.blue(msg), file=self.out, sep=sep, end=end)

    def print(self, msg: str, sep=' ', end='\n'):
        print(msg, file=self.out, sep=sep, end=end)

    @classmethod
    @property
    @abstractmethod
    def cms_name(cls) -> str:
        raise NotImplementedError

    @classmethod
    @property
    @abstractmethod
    def config_file(cls) -> Union[str, list[str]]:
        raise NotImplementedError

    @staticmethod
    @abstractmethod
    def is_config(path: Path) -> bool:
        """test if a path is a site config"""
        raise NotImplementedError

    @abstractmethod
    def scan_install(self, conf_path: Path):
        """called on each located installation"""
        raise NotImplementedError
