#! /opt/imh-python/bin/python3
''' Gather information about system environment '''

import os
import sys
import json
from argparse import ArgumentParser

__maintainer__ = "Daniel K"
__email__ = "danielk@inmotionhosting.com"
__version__ = "0.1.1"
__date__ = "2016-09-22"


# Globals
DATA = {"status": 'uninitiated', "version": __version__}


class EnvInfoError(Exception):
    '''Custom error class for module.'''

    def __init__(self, message):
        '''Init with call to base constructor.'''
        super().__init__(message)


def get_data(min_version=None, max_version=None):
    '''
    Main function for module, used to acquire environment information.

    min_version - If set, require envinfo.py
        to have a version greater than min_version

    max_version - If set, require envinfo.py
        to have a version less than max_version

    Returns dictionary of environment information. Currently, the only
    guaranteed entries are 'type', 'platform', and 'users'. The following
    are the keys that which may be present:

      type - General type of environment. Can be 'cpanel' or 'unknown'
      platform - Operating system information
      web_server - Type of webserver. Currently only 'apache'
      users - List of users.

      Apache related keys:

      easyapache - Version of EasyApache if present
      apache_conf - Location of httpd.conf for Apache
      apache_domlogs - Location of Apache domlogs

    On error will throw EnvInfoError.
    '''

    if min_version is not None and min_version > __version__:
        raise EnvInfoError(
            'Minimum version {} is greater than current version {}'.format(
                min_version, __version__
            )
        )

    if max_version is not None and max_version < __version__:
        raise EnvInfoError(
            'Maximum version {} is less than current version {}'.format(
                max_version, __version__
            )
        )

    if os.path.isdir('/var/cpanel'):
        DATA['type'] = 'cpanel'
    else:
        DATA['type'] = 'unknown'

    if os.path.isfile('/etc/apache2/conf/httpd.conf'):
        DATA['apache_conf'] = '/etc/apache2/conf/httpd.conf'
        DATA['apache_domlogs'] = '/var/log/apache2/domlogs'
        DATA['web_server'] = 'apache'
        if DATA['type'] == 'cpanel':
            DATA['easyapache'] = '4'
    elif os.path.isfile('/usr/local/apache/conf/httpd.conf'):
        DATA['apache_conf'] = '/usr/local/apache/conf/httpd.conf'
        DATA['apache_domlogs'] = '/usr/local/apache/domlogs'
        DATA['web_server'] = 'apache'
        if DATA['type'] == 'cpanel':
            DATA['easyapache'] = '3'

    user_list = []

    if DATA['type'] == 'cpanel':
        user_list = os.listdir('/var/cpanel/users/')
    elif os.path.isfile('/etc/passwd'):
        # If we don't know how else to find users, look to /etc/passwd
        if os.path.exists('/etc/passwd'):
            with open('/etc/passwd') as file_handle:
                try:
                    for line in file_handle:
                        uid = int(line.split(':')[2])
                        if 500 <= uid < 32000:
                            user_list.append(line.split(':')[0])
                except OSError as e:
                    # Catch, and throw back IOErrors.
                    raise EnvInfoError('Could not read from /etc/passwd') from e
    else:
        # No known way to find users
        raise EnvInfoError("Unable to make user list for server type.")

    DATA['users'] = user_list

    DATA['status'] = 'initiated'

    return DATA


def parse_args():
    '''
    Parse command line arguments
    '''

    parser = ArgumentParser(description=__doc__)

    output_group = parser.add_argument_group("Output options")

    output_type_group = output_group.add_mutually_exclusive_group()

    output_type_group.add_argument(
        "-e",
        "--environment",
        action='store_const',
        dest='output_type',
        const='env',
        help=(
            "Print general system environment information."
            "This is the defualt."
        ),
    )

    output_type_group.add_argument(
        "-u",
        "--users",
        action='store_const',
        dest='output_type',
        const='users',
        help="Print list of users rather than general environment.",
    )

    output_type_group.add_argument(
        "--apache-conf",
        action='store_const',
        dest='output_type',
        const='apache_conf',
        help="Print location of Apache's httpd.conf.",
    )

    output_group.add_argument(
        "-j", "--json", action='store_true', help="Print results in JSON"
    )

    parser.add_argument(
        "-v",
        "-V",
        "--version",
        action='store_true',
        help="Print version information and exit.",
    )

    args = parser.parse_args()

    if args.version:
        print(f"Environment Information tool version {__version__}")
        print(f"Last modified on {__date__}.")
        sys.exit(0)

    return (args.output_type, args.json)


def main():
    '''Function to be called when run from the command line'''

    (output_type, use_json) = parse_args()

    try:
        data = get_data()
    except EnvInfoError as error:
        print("Error:")
        print(error)

    if output_type == 'users':
        users = data['users']
        if use_json:

            print(
                json.dumps(
                    users, sort_keys=True, indent=4, separators=(',', ': ')
                )
            )

        else:
            print("\n".join(users))
    elif output_type == 'apache_conf':
        if 'apache_conf' not in data:
            sys.stderr.write('No Apache configuration found\n')
            sys.exit(255)
        print(data['apache_conf'])
    else:
        if use_json:

            print(
                json.dumps(
                    data, sort_keys=True, indent=4, separators=(',', ': ')
                )
            )

        else:
            print()
            print("Server Environment")
            print()
            print("Server Type:            {}".format(data['type']))
            print(
                "OS:                     {}".format(" ".join(data['platform']))
            )
            print("Web Server:             {}".format(data['web_server']))
            print("Apache configuration:   {}".format(data['apache_conf']))
            print("Apache domain logs:     {}".format(data['apache_domlogs']))
            print()
            print("Users:")
            print(" ".join(data['users']))
            print()


if __name__ == "__main__":
    main()
