#!/opt/imh-python/bin/python3
"""Removes Updraft backups older than 7 days"""
from os.path import getctime
from pathlib import Path
from collections import defaultdict
import rads
from guds_modules.base import ModuleBase

REGEX = r".*\/wp-content\/updraft\/backup.+(tar|tmp|gz|zip)$"


class Module(ModuleBase):
    """Removes Updraft backups older than 7 days"""

    def run_module(self, homedir):
        """This function locates updraft backups and handles their deletion"""
        backup_list = self.find(homedir, regex=REGEX, mtime="+7", type='f')
        if not backup_list:
            return {}  # no email
        # if total size is < 10GiB, do nothing
        if sum(map(self.calc_bytes, backup_list)) < 10*2**30:
            self.logger.debug('backups too small, skipping %s', homedir)
            return {}  # no email
        # group by parent directories
        parents: defaultdict[Path, list[Path]] = defaultdict(list)
        for backup in backup_list:
            parents[backup.parent].append(backup)
        for dir_items in parents.values():
            # in each directory, remove all but the latest backup
            if len(dir_items) >= 2:
                dir_items.remove(max(dir_items, key=getctime))
                self.delete_items(dir_items)
        if rads.IMH_CLASS == 'reseller':
            return {}  # no email
        return {
            'template': 379 if rads.IMH_CLASS == 'hub' else 104,
            'variable_1': ', '.join(map(str, parents)),
        }
