import pytest
import os
import sys
import logging
import yaml
from copy import deepcopy
from conftest_autosuspend import *

TESTDIR = os.path.dirname(os.path.realpath(__file__))
RADSDIR = os.path.abspath('%s/..' % TESTDIR)
sys.path.append(RADSDIR)
# pylint: disable=unused-import
from autosuspend import Autosuspend, get_sa_dict
import autosuspend


@pytest.mark.parametrize(
    ('sa_data', 'yaml_data', 'overloaded', 'users'),
    [
        (
            sa_data1,
            yaml_data1,
            True,
            (
                ('argosy6', 'suspension'),
                ('clyde', 'skip'),
                ('herrew5', 'warning'),
            ),
        ),
    ],
)
def test_delta_usage(
    tmpdir, tmpdir_config, monkeypatch, sa_data, yaml_data, overloaded, users
):
    monkeypatch.setattr("autosuspend.sh.sa", sa_data)
    monkeypatch.setattr("autosuspend.try_open_yaml", yaml_data)
    monkeypatch.setattr(
        'autosuspend.check_interval_file_age', lambda *x, **y: True
    )

    Autosuspend.config_files.append(tmpdir_config)

    a = Autosuspend()
    a.server_overloaded = overloaded
    a.logger.setLevel(logging.DEBUG)

    for user, outcome in users:
        assert a.test(a.users[user]) == outcome

    # baduser is not a cPanel user and shouldn't be among those considered
    assert 'baduser' not in a.users

    # Test parsing of prior suspensions from the suspension log
    # This should be pulled into its own test definition at some point
    # but i'm in a hurry

    # Because we're not able to generate a suspension log entry
    # for an autosuspension, seed the suspension log with one:
    datafile_fixture = tmpdir.mkdir('test').join('autosuspend.yaml')
    with open(a.suspension_log, 'a') as log:
        log.write(
            '2014-03-18:12:55:25 PDT py.test: argosy6 [SUSPENSION] ra perm '
            'root "AUTO SUSPENSION: Consumed 22cp within the last '
            'measured interval."' + "\n"
        )
    parsed_log = autosuspend.prior_events(
        log=a.suspension_log,
        log_offset=a.suspension_log_offset,
        data_file=str(datafile_fixture),
    )
    with open(str(datafile_fixture)) as data:
        unserialized_priors = yaml.load(data)

    assert 1395172525 in parsed_log['argosy6']['suspensions']
    assert len(parsed_log['herrew5']['warnings']) == 1
    assert dict(parsed_log.items()) == unserialized_priors


@pytest.mark.parametrize(
    ('load', 'ncores', 'factor', 'expected'),
    [
        ((0, 0, 90), 8, 1, True),
        ((0, 0, 11), 2, 4.5, True),
        ((0, 0, 22), 22, 1.5, False),
    ],
)
def test_overloaded(monkeypatch, load, ncores, factor, expected):
    monkeypatch.setattr("os.getloadavg", lambda: load)
    monkeypatch.setattr("autosuspend.cpu_count", lambda: ncores)
    assert autosuspend.server_overloaded(factor=factor) == expected


def test_existing_data(monkeypatch, tmpdir):

    prior_yaml = '%s/fixtures/existing_data_autosuspend.yaml' % TESTDIR
    with open(prior_yaml) as data_file:
        priors_test_data = yaml.load(data_file)

    offile = tmpdir.join('offset')
    outfile = tmpdir.join('yamlfile')

    monkeypatch.setattr(
        "autosuspend.try_open_yaml", lambda x: deepcopy(priors_test_data)
    )

    priors = autosuspend.prior_events(
        log='%s/fixtures/existing_data_suspension.log' % TESTDIR,
        log_offset=str(offile),
        data_file=str(outfile),
    )

    with open(str(outfile)) as data:
        unserialized_priors = yaml.load(data)

    assert priors != priors_test_data

    # The mailbu5 example will fail if run on a system not on Pacific time
    # The test should be fixed, but effort
    assert sorted(unserialized_priors['mailbu5']['warnings'])[-1] == 1395334684

    assert unserialized_priors['flirti5'] == {
        'suspensions': [],
        'warnings': [1395269890],
    }


@pytest.mark.skipif(os.geteuid != 0, reason="Must be run as root")
@pytest.mark.parametrize(
    ('sa_data,yaml_data,overloaded'),
    [
        (sa_data1, yaml_data1, True),
    ],
)
def test_full(reset, monkeypatch, sa_data, yaml_data, overloaded, user):
    b = Autosuspend()
    b.server_overloaded = overloaded
    assert a.run() == None
    for user in a.users:
        reset(user)
