#!/opt/imh-python/bin/python3
"""Runs check_software on all users and stores the results
in /var/log/software_reports"""
# Written by JosephM
from argparse import ArgumentParser
import os
import subprocess
import sys
import multiprocessing
import time
from pathlib import Path
import rads
from cpapis import whmapi1, CpAPIError

LOGDIR = Path("/var/log/software_reports")
MAX_LOAD = multiprocessing.cpu_count()

# users to not run check_software on
RESERVED = rads.OUR_RESELLERS + [rads.SECURE_USER]


def main():
    # blank argparse so -h doesn't surprise anyone by immediately running
    ArgumentParser(description=__doc__).parse_args()
    if os.getuid() != 0:
        print("This tool must be run as root. Exiting...")
        sys.exit(1)
    LOGDIR.mkdir(mode=0o755, parents=True, exist_ok=True)
    check_software_cmd = ['/opt/sharedrads/check_software', '-r']
    rcode = 0

    try:
        api = whmapi1('listaccts', check=True)
    except CpAPIError as exc:
        sys.exit(f'Unable to retrieve user list! {exc}')
    users = [x['user'] for x in api['data']['acct'] if x not in RESERVED]

    for user in users:
        user_log = LOGDIR / f"{user}.txt"
        cmd = check_software_cmd.copy()
        cmd.append(user)
        try:
            with open(user_log, 'wb') as file:
                user_log.chmod(0o600)
                subprocess.run(cmd, stdout=file, stderr=file, check=False)
        except Exception:
            print("Error running check_software on", user, file=sys.stderr)
            rcode = 2
        if os.getloadavg()[0] > MAX_LOAD:
            time.sleep(5)
            print("Sleeping due to high server load...")
            while os.getloadavg()[0] > MAX_LOAD:
                time.sleep(5)
    sys.exit(rcode)


if __name__ == '__main__':
    main()
